require('dotenv').config();
const path = require('path');
const express = require('express');
const session = require('express-session');
const helmet = require('helmet');
const csrf = require('csurf');
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
const Database = require('better-sqlite3');
const http = require('http');
const { Server } = require('socket.io');

const app = express();
const server = http.createServer(app);
const io = new Server(server, { cors: { origin: false } });

const PORT = process.env.PORT || 3000;
const BINANCE_BASE = process.env.BINANCE_PAY_BASE || 'https://bpay.binanceapi.com';
const CERT_SN = process.env.BINANCE_PAY_API_KEY || '';
const SECRET  = process.env.BINANCE_PAY_SECRET || '';

// ---------- DB ----------
const db = new Database(path.join(__dirname, 'aviator.db'));
db.pragma('journal_mode = WAL');

db.exec(`
CREATE TABLE IF NOT EXISTS users (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  username TEXT UNIQUE NOT NULL,
  password_hash TEXT NOT NULL,
  balance REAL NOT NULL DEFAULT 0,
  created_at INTEGER NOT NULL
);
CREATE TABLE IF NOT EXISTS rounds (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  started_at INTEGER NOT NULL,
  crash_at REAL,
  completed INTEGER NOT NULL DEFAULT 0
);
CREATE TABLE IF NOT EXISTS bets (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  round_id INTEGER NOT NULL,
  user_id INTEGER,
  slot TEXT,
  amount REAL NOT NULL,
  cashout_at REAL,
  win REAL,
  created_at INTEGER NOT NULL
);
CREATE TABLE IF NOT EXISTS topups (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  user_id INTEGER NOT NULL,
  merchantTradeNo TEXT NOT NULL UNIQUE,
  prepayId TEXT,
  amount REAL NOT NULL,
  currency TEXT NOT NULL,
  status TEXT NOT NULL,
  credited INTEGER NOT NULL DEFAULT 0,
  created_at INTEGER NOT NULL,
  paid_at INTEGER
);
`);

// --- tiny helpers for migrations ---
function hasColumn(table, col){
  try { return db.prepare(`PRAGMA table_info(${table})`).all().some(r=>r.name===col); } catch { return false; }
}
// Migrate older schemas
try {
  if (!hasColumn('bets','user_id')) {
    db.exec(`ALTER TABLE bets ADD COLUMN user_id INTEGER;`);
    if (hasColumn('bets','player_id')) {
      db.exec(`UPDATE bets SET user_id = player_id WHERE user_id IS NULL;`);
    }
  }
  if (!hasColumn('bets','slot'))       db.exec(`ALTER TABLE bets ADD COLUMN slot TEXT;`);
  if (!hasColumn('bets','cashout_at')) db.exec(`ALTER TABLE bets ADD COLUMN cashout_at REAL;`);
  if (!hasColumn('bets','win'))        db.exec(`ALTER TABLE bets ADD COLUMN win REAL;`);
} catch(e){ console.error('DB migration error:', e); }

// --- prepared statements ---
const stmtCreateUser   = db.prepare('INSERT INTO users(username, password_hash, balance, created_at) VALUES (?,?,?,?)');
const stmtUserByName   = db.prepare('SELECT * FROM users WHERE username=?');
const stmtUserById     = db.prepare('SELECT * FROM users WHERE id=?');
const stmtUpdateBalance= db.prepare('UPDATE users SET balance=? WHERE id=?');

const stmtNewRound     = db.prepare('INSERT INTO rounds(started_at) VALUES (?)');
const stmtCompleteRound= db.prepare('UPDATE rounds SET crash_at=?, completed=1 WHERE id=?');
const stmtLast20       = db.prepare('SELECT crash_at FROM rounds WHERE completed=1 ORDER BY id DESC LIMIT 20');

const stmtInsertBet    = db.prepare('INSERT INTO bets (round_id, user_id, slot, amount, created_at) VALUES (?,?,?,?,?)');
const stmtCashout      = db.prepare('UPDATE bets SET cashout_at=?, win=? WHERE id=?');
const stmtUserBetsInRound = db.prepare('SELECT * FROM bets WHERE round_id=? AND user_id=?');

const stmtInsertTopup = db.prepare('INSERT INTO topups(user_id, merchantTradeNo, amount, currency, status, created_at) VALUES (?,?,?,?,?,?)');
const stmtUpdateTopupPrepay = db.prepare('UPDATE topups SET prepayId=?, status=? WHERE merchantTradeNo=?');
const stmtFindTopupByPrepay = db.prepare('SELECT * FROM topups WHERE prepayId=?');
const stmtFindTopupByTradeNo= db.prepare('SELECT * FROM topups WHERE merchantTradeNo=?');
const stmtMarkTopupPaid     = db.prepare('UPDATE topups SET status=?, paid_at=?, credited=1 WHERE merchantTradeNo=?');

// ---------- Security & Middleware ----------
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      scriptSrc:  ["'self'"],
      styleSrc:   ["'self'"],
      imgSrc:     ["'self'", 'data:', 'https:'], // QR images
      connectSrc: ["'self'", 'ws:', 'wss:'],
      objectSrc:  ["'none'"],
      baseUri:    ["'self'"],
      frameAncestors: ["'self'"]
    }
  },
  crossOriginEmbedderPolicy: false
}));

app.use(express.json());
app.use(express.urlencoded({ extended: false }));

const sessMiddleware = session({
  name: 'sid',
  secret: process.env.SESSION_SECRET || crypto.randomBytes(32).toString('hex'),
  resave: false,
  saveUninitialized: false,
  cookie: { httpOnly: true, sameSite: 'lax' }
});
app.use(sessMiddleware);
io.use((socket, next)=> sessMiddleware(socket.request, {}, next));

const csrfMiddleware = csrf();
app.use(csrfMiddleware);

// Rate-limit (very small, in-memory)
const rl = new Map();
function allow(key, limit=8, windowMs=2000){
  const now = Date.now();
  const rec = rl.get(key) || { count:0, reset: now + windowMs };
  if (now > rec.reset){ rec.count = 0; rec.reset = now + windowMs; }
  rec.count++; rl.set(key, rec);
  return rec.count <= limit;
}

// Static
app.use(express.static(path.join(__dirname, 'public'), { etag: true, maxAge: '1h' }));

// ---------- Auth helpers ----------
function requireAuth(req, res, next){ if (req.session.userId) return next(); return res.status(401).json({ error:'UNAUTH' }); }

// Pages (no inline scripts/styles in these files)
app.get('/', (req,res)=> res.sendFile(path.join(__dirname,'public','index.html')));
app.get('/login', (req,res)=> res.sendFile(path.join(__dirname,'public','login.html')));
app.get('/register', (req,res)=> res.sendFile(path.join(__dirname,'public','register.html')));
app.get('/topup', (req,res)=> req.session.userId ? res.sendFile(path.join(__dirname,'public','topup.html')) : res.redirect('/login'));

// ---------- API (Auth) ----------
app.get('/api/me', (req,res)=>{
  const user = req.session.userId ? stmtUserById.get(req.session.userId) : null;
  res.json({ user: user ? { id:user.id, username:user.username, balance:Number(user.balance.toFixed(2)) } : null, csrfToken: req.csrfToken() });
});

app.post('/api/register', (req,res)=>{
  try{
    if(!allow(req.ip)) return res.status(429).json({error:'Too many requests'});
    const u = String(req.body.username||'').trim();
    const p = String(req.body.password||'');
    if (u.length < 3 || p.length < 6) return res.status(400).json({error:'Login >=3, Parol >=6'});
    const hash = bcrypt.hashSync(p, 10);
    const info = stmtCreateUser.run(u, hash, 10000, Date.now());
    req.session.userId = info.lastInsertRowid;
    res.json({ ok:true });
  }catch(e){
    if (String(e).includes('UNIQUE')) return res.status(400).json({error:'Login band'});
    res.status(500).json({error:'SERVER'});
  }
});

app.post('/api/login', (req,res)=>{
  try{
    if(!allow(req.ip)) return res.status(429).json({error:'Too many requests'});
    const u = String(req.body.username||'').trim();
    const p = String(req.body.password||'');
    const row = stmtUserByName.get(u);
    if (!row) return res.status(400).json({error:'Noto\'g\'ri login/parol'});
    if (!bcrypt.compareSync(p, row.password_hash)) return res.status(400).json({error:'Noto\'g\'ri login/parol'});
    req.session.userId = row.id;
    res.json({ ok:true });
  }catch(e){ res.status(500).json({error:'SERVER'}); }
});

app.post('/api/logout', requireAuth, (req,res)=>{
  req.session.destroy(()=> res.json({ok:true}));
});

// ---------- Binance Pay helpers ----------
function randNonce32(){
  return crypto.randomBytes(24).toString('base64').replace(/[^a-zA-Z0-9]/g,'').slice(0,32);
}
function signHeaders(bodyObj){
  const body = JSON.stringify(bodyObj);
  const ts = Date.now().toString();
  const nonce = randNonce32();
  const payload = `${ts}\n${nonce}\n${body}\n`;
  const sig = crypto.createHmac('sha512', SECRET).update(payload).digest('hex').toUpperCase();
  return {
    'content-type': 'application/json',
    'BinancePay-Timestamp': ts,
    'BinancePay-Nonce': nonce,
    'BinancePay-Certificate-SN': CERT_SN,
    'BinancePay-Signature': sig
  };
}

// Create order
app.post('/api/pay/create', requireAuth, async (req,res)=>{
  try{
    if(!allow(req.ip)) return res.status(429).json({error:'Too many requests'});
    const amount = Number(req.body?.amount);
    const currency = String(req.body?.currency||'USDT').toUpperCase();
    if(!Number.isFinite(amount) || amount < 0.01) return res.status(400).json({error:'Min 0.01'});
    if(!['USDT','USDC'].includes(currency)) return res.status(400).json({error:'Currency USDT/USDC'});

    const u = stmtUserById.get(req.session.userId);
    const merchantTradeNo = `TOPUP-${u.id}-${Date.now()}-${Math.floor(Math.random()*1e6)}`;
    stmtInsertTopup.run(u.id, merchantTradeNo, amount, currency, 'INITIAL', Date.now());

    const body = {
      env: { terminalType: 'WEB' },
      merchantTradeNo,
      orderAmount: Number(amount.toFixed(8)),
      currency,
      description: 'Wallet top-up',
      goodsDetails: [{ goodsType: '02', goodsCategory:'6000', referenceGoodsId:'WALLET_TOPUP', goodsName:'Balance top-up' }]
    };

    const headers = signHeaders(body);
    const r = await fetch(`${BINANCE_BASE}/binancepay/openapi/v3/order`, { method:'POST', headers, body: JSON.stringify(body) });
    const j = await r.json();
    if(j.status!=='SUCCESS') return res.status(400).json({error:j.errorMessage||'BINANCE_FAIL', raw:j});

    stmtUpdateTopupPrepay.run(j.data.prepayId, 'PENDING', merchantTradeNo);
    res.json({ merchantTradeNo, prepayId:j.data.prepayId, qrcodeLink:j.data.qrcodeLink, qrContent:j.data.qrContent, checkoutUrl:j.data.checkoutUrl, universalUrl:j.data.universalUrl });
  }catch(e){ res.status(500).json({error:'SERVER', detail:String(e)}); }
});

// Query status + credit balance
app.get('/api/pay/status', requireAuth, async (req,res)=>{
  try{
    const prepayId = req.query.prepayId;
    const merchantTradeNo = req.query.merchantTradeNo;
    const body = prepayId ? { prepayId } : { merchantTradeNo };
    const headers = signHeaders(body);

    const r = await fetch(`${BINANCE_BASE}/binancepay/openapi/v2/order/query`, { method:'POST', headers, body: JSON.stringify(body) });
    const j = await r.json();
    if(j.status!=='SUCCESS') return res.status(400).json({error:j.errorMessage||'QUERY_FAIL', raw:j});

    const data = j.data;
    if (data.status === 'PAID'){
      const row = prepayId ? stmtFindTopupByPrepay.get(prepayId) : stmtFindTopupByTradeNo.get(merchantTradeNo);
      if (row && !row.credited){
        const u = stmtUserById.get(row.user_id);
        const paid = Number.parseFloat(data.orderAmount || row.amount);
        const newBal = Number((u.balance + paid).toFixed(2));
        stmtUpdateBalance.run(newBal, u.id);
        stmtMarkTopupPaid.run('PAID', Date.now(), row.merchantTradeNo);
      }
    }
    res.json({ status: data.status, orderAmount: data.orderAmount, currency: data.currency, transactionId: data.transactionId || null });
  }catch(e){ res.status(500).json({error:'SERVER', detail:String(e)}); }
});

// ---------- Game Loop ----------
const PHASES = { BETTING:'betting', FLYING:'flying', CRASHED:'crashed' };
const BETTING_TIME = 7000; // ms
const TICK_MS = 100;       // ms
const GHOST_TARGET = 100;

let roundId = 0;
let phase = PHASES.BETTING;
let multiplier = 1.0;
let crashAt = 1.0;
let bettingEndsAt = 0;
let tickTimer = null;

// Per-user per-slot state
let betsMap = new Map();   // userId -> { A: {betId, amount}, B: {betId, amount} }
let cashedMap = new Map(); // userId -> { A: {at, win, betId}, B: {at, win, betId} }

// Ghosts
let ghosts = [];
function randName(){
  const syll = ['al','bo','chi','do','er','fa','gi','ha','io','jo','ka','li','mo','na','or','pa','qi','ru','so','ta','ur','vi','wo','xa','ya','zo'];
  const s = () => syll[Math.floor(Math.random()*syll.length)];
  const n = (s()+s()+s());
  return n.charAt(0).toUpperCase()+n.slice(1);
}
function maskName(name){ if(!name) return '--'; if(name.length<2) return name; return `${name[0]}...${name[name.length-1]}`; }
function randBet(){ return Math.round((1 + Math.random()*50) * 100) / 100; }
function randCash(){ return Math.max(1.05, Math.round((1.1 + Math.random()*4)*100)/100); }
function prepareGhosts(){
  const realCount = Array.from(io.sockets.sockets.values()).filter(s=>s.data.user).length;
  const need = Math.max(0, GHOST_TARGET - realCount);
  ghosts = Array.from({length: need}, () => {
    const name = randName();
    const hasA = Math.random() < 0.7; const hasB = Math.random() < 0.5; const useA = hasA || (!hasA && !hasB); const useB = hasB;
    return {
      name, mask: maskName(name),
      betA: useA ? randBet() : 0, cashAtA: useA ? randCash() : 0, cashedAtA:null, winA:null,
      betB: useB ? randBet() : 0, cashAtB: useB ? randCash() : 0, cashedAtB:null, winB:null
    };
  });
}

function randomCrash(){
  // Heavy-tailed-ish distribution
  const r = Math.random();
  if (r < 0.10) return 1.0 + Math.random()*0.2;     // early crash 1.00-1.20
  if (r < 0.55) return 1.2 + Math.random()*1.5;     // 1.2 - 2.7
  if (r < 0.85) return 2.7 + Math.random()*2.5;     // 2.7 - 5.2
  return 5.2 + Math.random()*10;                    // 5.2 - 15.2
}

function broadcast(){
  const players = [];
  for (const [, sock] of io.sockets.sockets){
    const u = sock.data.user; if(!u) continue;
    const b = betsMap.get(u.id) || {};
    const c = cashedMap.get(u.id) || {};
    players.push({
      uid: u.id, name: maskName(u.username), balance: Number(u.balance.toFixed(2)), isBot:false,
      betA: b.A?.amount || 0, cashedAtA: c.A?.at || null, winA: c.A?.win || null,
      betB: b.B?.amount || 0, cashedAtB: c.B?.at || null, winB: c.B?.win || null,
    });
  }
  for (const g of ghosts){
    players.push({
      uid: -1, name: g.mask, balance: 0, isBot:true,
      betA: g.betA||0, cashedAtA: g.cashedAtA, winA: g.winA,
      betB: g.betB||0, cashedAtB: g.cashedAtB, winB: g.winB,
    });
  }
  io.emit('state', {
    roundId, phase, multiplier, crashAt: phase===PHASES.CRASHED?crashAt:null,
    bettingEndsAt, serverTime: Date.now(), players
  });
}

function resetRound(){
  roundId = stmtNewRound.run(Date.now()).lastInsertRowid;
  phase = PHASES.BETTING; multiplier = 1.0; crashAt = randomCrash();
  bettingEndsAt = Date.now() + BETTING_TIME;
  betsMap.clear(); cashedMap.clear(); prepareGhosts();
  broadcast();
}

function startFlying(){
  phase = PHASES.FLYING; multiplier = 1.0; broadcast();
}

function crash(){
  phase = PHASES.CRASHED; multiplier = Number(multiplier.toFixed(2));
  stmtCompleteRound.run(multiplier, roundId);
  broadcast();
  setTimeout(resetRound, 3000);
}

function gameTick(){
  if (phase === PHASES.BETTING){
    if (Date.now() >= bettingEndsAt) startFlying();
  } else if (phase === PHASES.FLYING){
    // grow multiplier (approx)
    multiplier *= 1.012; // ~1.2% per tick
    if (multiplier >= crashAt) { crash(); return; }

    // Ghost auto-cash
    for (const g of ghosts){
      if (g.betA && !g.cashedAtA && multiplier >= g.cashAtA){ g.cashedAtA = Number(multiplier.toFixed(2)); g.winA = Number((g.betA*multiplier).toFixed(2)); }
      if (g.betB && !g.cashedAtB && multiplier >= g.cashAtB){ g.cashedAtB = Number(multiplier.toFixed(2)); g.winB = Number((g.betB*multiplier).toFixed(2)); }
    }
    broadcast();
  }
}

tickTimer = setInterval(gameTick, TICK_MS);
resetRound();

// ---------- Socket.io ----------
io.on('connection', (socket)=>{
  const req = socket.request;
  const sid = req.session;
  if (!sid || !sid.userId){ socket.disconnect(true); return; }
  const u = stmtUserById.get(sid.userId);
  if(!u){ socket.disconnect(true); return; }
  socket.data.user = { id:u.id, username:u.username, balance:u.balance };
  socket.emit('hello', { id:u.id, username:u.username, balance:Number(u.balance.toFixed(2)) });
  broadcast();

  socket.on('placeBet', (payload)=>{
    if (!allow(socket.id)) return socket.emit('errorMsg','Rate limited');
    if (phase !== PHASES.BETTING) return socket.emit('errorMsg','Betting yopilgan');

    const slot = String(payload?.slot||'').toUpperCase();
    let amount = Number(payload?.amount);
    if (!['A','B'].includes(slot)) return socket.emit('errorMsg','Noto\'g\'ri slot');
    if (!Number.isFinite(amount) || amount<=0) return socket.emit('errorMsg','Noto\'g\'ri summa');

    const fresh = stmtUserById.get(socket.data.user.id);
    const userBets = betsMap.get(fresh.id) || {};
    if (userBets[slot]) return socket.emit('errorMsg',`Slot ${slot} uchun bet bor`);
    if (amount > fresh.balance) return socket.emit('errorMsg','Balans yetarli emas');

    const info = stmtInsertBet.run(roundId, fresh.id, slot, amount, Date.now());
    const newBal = Number((fresh.balance - amount).toFixed(2));
    stmtUpdateBalance.run(newBal, fresh.id);
    userBets[slot] = { betId: info.lastInsertRowid, amount };
    betsMap.set(fresh.id, userBets);
    socket.data.user.balance = newBal;
    broadcast();
  });

  socket.on('cashOut', (payload)=>{
    if (!allow(socket.id)) return socket.emit('errorMsg','Rate limited');
    if (phase !== PHASES.FLYING) return socket.emit('errorMsg','Hali uchish emas');

    const slot = String(payload?.slot||'').toUpperCase();
    if (!['A','B'].includes(slot)) return socket.emit('errorMsg','Noto\'g\'ri slot');

    const fresh = stmtUserById.get(socket.data.user.id);
    const b = (betsMap.get(fresh.id)||{})[slot];
    if(!b) return socket.emit('errorMsg',`Slot ${slot} uchun bet yo'q`);

    const cashed = cashedMap.get(fresh.id) || {};
    if (cashed[slot]) return socket.emit('errorMsg',`Slot ${slot} allaqachon cash out`);

    const win = Number((b.amount * multiplier).toFixed(2));
    stmtCashout.run(Number(multiplier.toFixed(2)), win, b.betId);
    const newBal = Number((fresh.balance + win).toFixed(2));
    stmtUpdateBalance.run(newBal, fresh.id);
    cashed[slot] = { at: Number(multiplier.toFixed(2)), win, betId: b.betId };
    cashedMap.set(fresh.id, cashed);
    socket.data.user.balance = newBal;
    broadcast();
  });

  socket.on('disconnect', ()=>{});
});

// ---------- Start ----------
server.listen(PORT, ()=> console.log(`Aviator running on http://localhost:${PORT}`));